// Tutoring Website - Version 3 (HTML/CSS/JS)

// Subjects data
const subjects = [
  { 
    id: 'math', 
    name: 'Математика', 
    icon: 'icons/subject-math.svg', 
    color: '#F97316',
    description: 'Алгебра, геометрия, подготовка к ЕГЭ'
  },
  { 
    id: 'physics', 
    name: 'Физика', 
    icon: 'icons/subject-physics.svg', 
    color: '#FB923C',
    description: 'Механика, оптика, термодинамика'
  },
  { 
    id: 'english', 
    name: 'Английский', 
    icon: 'icons/subject-english.svg', 
    color: '#FDBA74',
    description: 'Грамматика, разговорная речь, подготовка к экзаменам'
  },
  { 
    id: 'russian', 
    name: 'Русский', 
    icon: 'icons/subject-russian.svg', 
    color: '#F97316',
    description: 'Орфография, пунктуация, сочинения'
  }
]

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'history'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initHistoryPage()
  }
  
  initSlider()
}

// Initialize history page
function initHistoryPage() {
  loadLessons()
  updateStats()
  
  // Tab switching
  const tabs = document.querySelectorAll('.tab')
  tabs.forEach(tab => {
    tab.addEventListener('click', function() {
      tabs.forEach(t => t.classList.remove('active'))
      this.classList.add('active')
      const tabType = this.dataset.tab
      renderLessons(tabType)
    })
  })
  
  // Initial render
  renderLessons('active')
  
  // Poll for storage changes
  setInterval(() => {
    loadLessons()
    updateStats()
  }, 500)
}

// Load lessons from localStorage
function loadLessons() {
  const saved = localStorage.getItem('tutoringLessons')
  let lessons = { active: [], past: [] }
  
  if (saved) {
    try {
      lessons = JSON.parse(saved)
    } catch (e) {
      lessons = { active: [], past: [] }
    }
  } else {
    // Default lessons
    lessons = {
      active: [
        {
          id: 1,
          subject: 'Математика',
          tutor: 'Анна Петрова',
          date: '2025-03-28',
          time: '15:00',
          status: 'active'
        },
        {
          id: 2,
          subject: 'Физика',
          tutor: 'Иван Сидоров',
          date: '2025-03-29',
          time: '16:30',
          status: 'active'
        }
      ],
      past: [
        {
          id: 3,
          subject: 'Математика',
          tutor: 'Анна Петрова',
          date: '2025-03-20',
          time: '15:00',
          efficiency: 5
        },
        {
          id: 4,
          subject: 'Английский',
          tutor: 'Мария Козлова',
          date: '2025-03-18',
          time: '14:00',
          efficiency: 4
        },
        {
          id: 5,
          subject: 'Русский',
          tutor: 'Елена Смирнова',
          date: '2025-03-15',
          time: '13:00',
          efficiency: 5
        }
      ]
    }
    localStorage.setItem('tutoringLessons', JSON.stringify(lessons))
  }
  
  return lessons
}

// Render lessons
function renderLessons(type) {
  const lessons = loadLessons()
  const currentLessons = lessons[type]
  const lessonsGrid = document.getElementById('lessonsGrid')
  
  if (!lessonsGrid) return
  
  if (currentLessons.length === 0) {
    lessonsGrid.innerHTML = `
      <div class="empty-state">
        <img src="icons/empty-book.svg" alt="" class="empty-icon" width="72" height="72" />
        <p>${type === 'active' ? 'У вас нет активных уроков' : 'История уроков пуста'}</p>
      </div>
    `
    return
  }
  
  lessonsGrid.innerHTML = currentLessons.map(lesson => {
    const stars = lesson.efficiency ? renderStars(lesson.efficiency) : ''
    return `
                    <div class="lesson-card">
                        <div class="lesson-header">
                          <div class="lesson-icon-wrapper">
                            <img src="icons/lesson-book.svg" alt="" class="lesson-icon" width="36" height="36" />
                          </div>
                          <div class="lesson-info">
                            <h3 class="lesson-subject">${lesson.subject}</h3>
                            <p class="lesson-tutor">${lesson.tutor}</p>
                          </div>
                        </div>
        <div class="lesson-details">
          <div class="lesson-detail">
            <img src="icons/detail-calendar.svg" alt="" class="detail-icon" width="20" height="20" />
            <span>${new Date(lesson.date).toLocaleDateString('ru-RU', {
              day: 'numeric',
              month: 'long',
              year: 'numeric'
            })}</span>
          </div>
          <div class="lesson-detail">
            <img src="icons/detail-clock.svg" alt="" class="detail-icon" width="20" height="20" />
            <span>${lesson.time}</span>
          </div>
        </div>
        ${lesson.status === 'active' ? `
          <div class="lesson-status active-badge">Запланирован</div>
        ` : ''}
        ${lesson.efficiency ? `
          <div class="lesson-efficiency">
            <div class="efficiency-label">
              <img src="icons/efficiency-trend.svg" alt="" class="efficiency-icon" width="18" height="18" />
              <span>Эффективность:</span>
            </div>
            <div class="stars-wrapper">
              ${stars}
            </div>
          </div>
        ` : ''}
      </div>
    `
  }).join('')
}

function renderStars(rating) {
  return Array.from({ length: 5 }, (_, i) => {
    const filled = i < rating
    return `<img src="icons/star-${filled ? 'filled' : 'empty'}.svg" alt="" class="star" width="18" height="18" style="filter: ${filled ? 'none' : 'opacity(0.3)'}" />`
  }).join('')
}

// Update stats
function updateStats() {
  const lessons = loadLessons()
  const stats = [
    { 
      value: lessons.past.length, 
      label: 'Пройдено уроков',
      icon: 'icons/stat-book.svg',
      color: '#F97316'
    },
    { 
      value: new Set(lessons.past.map(l => l.tutor)).size, 
      label: 'Репетиторов',
      icon: 'icons/stat-user.svg',
      color: '#FB923C'
    },
    { 
      value: lessons.past.length > 0 
        ? Math.round(lessons.past.reduce((sum, l) => sum + (l.efficiency || 0), 0) / lessons.past.length * 10) / 10
        : 0, 
      label: 'Средняя оценка',
      icon: 'icons/stat-star.svg',
      color: '#FDBA74'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card-vertical">
        <div class="stat-icon-wrapper" style="background: ${stat.color}20">
          <img src="${stat.icon}" alt="" class="stat-icon" width="28" height="28" />
        </div>
        <div class="stat-content-vertical">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
  
  // Update tab counts
  const activeCount = document.getElementById('activeCount')
  const pastCount = document.getElementById('pastCount')
  if (activeCount) activeCount.textContent = lessons.active.length
  if (pastCount) pastCount.textContent = lessons.past.length
}

// Initialize book page
function initBookPage() {
  let selectedSubject = null
  
  const subjectsGrid = document.getElementById('subjectsGrid')
  const dateInput = document.getElementById('date')
  const timeSelect = document.getElementById('time')
  const bookingForm = document.getElementById('bookingForm')
  const successMessage = document.getElementById('successMessage')
  
  // Set date limits
  const today = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
  if (dateInput) {
    dateInput.min = today
    dateInput.max = maxDate
  }
  
  // Render subjects
  if (subjectsGrid) {
    subjectsGrid.innerHTML = subjects.map(subject => `
      <button 
        type="button" 
        class="subject-card" 
        data-subject="${subject.id}"
        style="border-color: rgba(249, 115, 22, 0.2); background: rgba(255, 255, 255, 0.9);"
      >
        <div class="subject-icon-wrapper" style="background: ${subject.color}15">
          <img src="${subject.icon}" alt="${subject.name}" width="32" height="32" />
        </div>
        <div class="subject-content">
          <h3 class="subject-name">${subject.name}</h3>
          <p class="subject-description">${subject.description}</p>
        </div>
      </button>
    `).join('')
    
    // Subject selection
    subjectsGrid.querySelectorAll('.subject-card').forEach(card => {
      card.addEventListener('click', function() {
        subjectsGrid.querySelectorAll('.subject-card').forEach(c => {
          c.classList.remove('selected')
          c.style.borderColor = 'rgba(249, 115, 22, 0.2)'
          c.style.background = 'rgba(255, 255, 255, 0.9)'
        })
        this.classList.add('selected')
        selectedSubject = this.dataset.subject
        const subject = subjects.find(s => s.id === selectedSubject)
        if (subject) {
          this.style.borderColor = subject.color
          this.style.background = `rgba(249, 115, 22, 0.05)`
        }
        updateSubmitButton()
      })
    })
  }
  
  function updateSubmitButton() {
    const submitBtn = document.querySelector('.submit-btn')
    if (submitBtn) {
      const isValid = selectedSubject && dateInput?.value && timeSelect?.value
      submitBtn.disabled = !isValid
    }
  }
  
  if (dateInput) {
    dateInput.addEventListener('change', updateSubmitButton)
  }
  
  if (timeSelect) {
    timeSelect.addEventListener('change', updateSubmitButton)
  }
  
  // Form submission
  if (bookingForm) {
    bookingForm.addEventListener('submit', function(e) {
      e.preventDefault()
      
      if (!selectedSubject || !dateInput?.value || !timeSelect?.value) {
        return
      }
      
      const subjectName = subjects.find(s => s.id === selectedSubject).name
      const tutors = {
        'math': 'Анна Петрова',
        'physics': 'Иван Сидоров',
        'english': 'Мария Козлова',
        'russian': 'Елена Смирнова'
      }
      
      const lessons = loadLessons()
      const newLesson = {
        id: Date.now(),
        subject: subjectName,
        tutor: tutors[selectedSubject],
        date: dateInput.value,
        time: timeSelect.value,
        status: 'active'
      }
      
      lessons.active.push(newLesson)
      localStorage.setItem('tutoringLessons', JSON.stringify(lessons))
      
      // Show success message
      if (successMessage) {
        successMessage.style.display = 'flex'
        setTimeout(() => {
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

// Initialize slider
function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return
  
  const slides = slider.querySelectorAll('.slide')
  const dots = slider.querySelectorAll('.dot')
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')
  
  let currentIndex = 0
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
  }
  
  function nextSlide() {
    currentIndex = (currentIndex + 1) % slides.length
    showSlide(currentIndex)
  }
  
  function prevSlide() {
    currentIndex = (currentIndex - 1 + slides.length) % slides.length
    showSlide(currentIndex)
  }
  
  if (prevBtn) prevBtn.addEventListener('click', prevSlide)
  if (nextBtn) nextBtn.addEventListener('click', nextSlide)
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentIndex = index
      showSlide(currentIndex)
    })
  })
  
  // Auto-advance
  setInterval(nextSlide, 3000)
  
  // Initial show
  showSlide(0)
}

// Initialize on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initPage)
} else {
  initPage()
}
